<?php
class Message {
    // Connexion à la base de données et nom de la table
    private $conn;
    private $table_name = "messages";
    private $encryption_key = "S3cur3M3ss3ng3rK3y2025!"; // Clé de chiffrement
    
    // Propriétés de l'objet
    public $id;
    public $sender_id;
    public $receiver_id;
    public $message;
    public $is_read;
    public $created_at;
    
    // Constructeur avec $db comme connexion à la base de données
    public function __construct($db) {
        $this->conn = $db;
    }
    
    // Méthode pour chiffrer un message
    private function encrypt($message) {
        $iv = openssl_random_pseudo_bytes(16);
        $encrypted = openssl_encrypt($message, 'AES-256-CBC', $this->encryption_key, 0, $iv);
        return base64_encode($iv . $encrypted);
    }
    
    // Méthode pour déchiffrer un message
    private function decrypt($encryptedMessage) {
        $data = base64_decode($encryptedMessage);
        $iv = substr($data, 0, 16);
        $encrypted = substr($data, 16);
        return openssl_decrypt($encrypted, 'AES-256-CBC', $this->encryption_key, 0, $iv);
    }
    
    // Créer un nouveau message
    public function create() {
        // Requête d'insertion
        $query = "INSERT INTO " . $this->table_name . "
                  SET sender_id = :sender_id,
                      receiver_id = :receiver_id,
                      message = :message,
                      is_read = 0,
                      created_at = NOW()";
        
        // Préparer la requête
        $stmt = $this->conn->prepare($query);
        
        // Nettoyer et chiffrer le message
        $clean_message = htmlspecialchars(strip_tags($this->message));
        $encrypted_message = $this->encrypt($clean_message);
        
        // Lier les valeurs
        $stmt->bindParam(":sender_id", $this->sender_id);
        $stmt->bindParam(":receiver_id", $this->receiver_id);
        $stmt->bindParam(":message", $encrypted_message);
        
        // Exécuter la requête
        if ($stmt->execute()) {
            return true;
        }
        
        return false;
    }
    
    // Récupérer la conversation entre deux utilisateurs
    public function getConversation($user1_id, $user2_id) {
        // Requête pour récupérer les messages
        $query = "SELECT m.id, m.sender_id, m.receiver_id, m.message, m.is_read, 
                         DATE_FORMAT(m.created_at, '%H:%i') as created_at
                  FROM " . $this->table_name . " m
                  WHERE (m.sender_id = :user1_id AND m.receiver_id = :user2_id)
                     OR (m.sender_id = :user2_id AND m.receiver_id = :user1_id)
                  ORDER BY m.created_at ASC";
        
        // Préparer la requête
        $stmt = $this->conn->prepare($query);
        
        // Lier les valeurs
        $stmt->bindParam(":user1_id", $user1_id);
        $stmt->bindParam(":user2_id", $user2_id);
        
        // Exécuter la requête
        $stmt->execute();
        
        // Tableau de messages
        $messages = array();
        
        // Récupérer les résultats
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            // Déchiffrer le message
            $decrypted_message = $this->decrypt($row['message']);
            
            $message = array(
                "id" => $row['id'],
                "sender_id" => $row['sender_id'],
                "receiver_id" => $row['receiver_id'],
                "message" => $decrypted_message,
                "is_read" => $row['is_read'],
                "created_at" => $row['created_at']
            );
            
            $messages[] = $message;
        }
        
        return $messages;
    }
    
    // Marquer les messages comme lus
    public function markAsRead($sender_id, $receiver_id) {
        // Requête de mise à jour
        $query = "UPDATE " . $this->table_name . "
                  SET is_read = 1
                  WHERE sender_id = :sender_id AND receiver_id = :receiver_id AND is_read = 0";
        
        // Préparer la requête
        $stmt = $this->conn->prepare($query);
        
        // Lier les valeurs
        $stmt->bindParam(":sender_id", $sender_id);
        $stmt->bindParam(":receiver_id", $receiver_id);
        
        // Exécuter la requête
        if ($stmt->execute()) {
            return true;
        }
        
        return false;
    }
    
    // Récupérer le nombre de messages non lus
    public function getUnreadCount($receiver_id, $sender_id) {
        // Requête pour compter les messages non lus
        $query = "SELECT COUNT(*) as unread_count
                  FROM " . $this->table_name . "
                  WHERE sender_id = :sender_id AND receiver_id = :receiver_id AND is_read = 0";
        
        // Préparer la requête
        $stmt = $this->conn->prepare($query);
        
        // Lier les valeurs
        $stmt->bindParam(":sender_id", $sender_id);
        $stmt->bindParam(":receiver_id", $receiver_id);
        
        // Exécuter la requête
        $stmt->execute();
        
        // Récupérer le résultat
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        
        return $row['unread_count'];
    }
    
    // Récupérer les nouveaux messages
    public function getNewMessages($user_id, $contact_id, $last_id) {
        // Requête pour récupérer les nouveaux messages
        $query = "SELECT m.id, m.sender_id, m.receiver_id, m.message, m.is_read, 
                         DATE_FORMAT(m.created_at, '%H:%i') as created_at
                  FROM " . $this->table_name . " m
                  WHERE ((m.sender_id = :user_id AND m.receiver_id = :contact_id)
                     OR (m.sender_id = :contact_id AND m.receiver_id = :user_id))
                     AND m.id > :last_id
                  ORDER BY m.created_at ASC";
        
        // Préparer la requête
        $stmt = $this->conn->prepare($query);
        
        // Lier les valeurs
        $stmt->bindParam(":user_id", $user_id);
        $stmt->bindParam(":contact_id", $contact_id);
        $stmt->bindParam(":last_id", $last_id);
        
        // Exécuter la requête
        $stmt->execute();
        
        // Tableau de messages
        $messages = array();
        
        // Récupérer les résultats
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            // Déchiffrer le message
            $decrypted_message = $this->decrypt($row['message']);
            
            $message = array(
                "id" => $row['id'],
                "sender_id" => $row['sender_id'],
                "receiver_id" => $row['receiver_id'],
                "message" => $decrypted_message,
                "is_read" => $row['is_read'],
                "created_at" => $row['created_at']
            );
            
            $messages[] = $message;
        }
        
        return $messages;
    }
    
    // Supprimer un message (pour l'utilisateur courant uniquement)
    public function deleteMessage($message_id, $user_id) {
        // Vérifier si l'utilisateur est autorisé à supprimer ce message
        $query = "SELECT sender_id FROM " . $this->table_name . " WHERE id = :message_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":message_id", $message_id);
        $stmt->execute();
        
        if ($stmt->rowCount() > 0) {
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Si l'utilisateur est l'expéditeur, il peut supprimer le message
            if ($row['sender_id'] == $user_id) {
                $delete_query = "DELETE FROM " . $this->table_name . " WHERE id = :message_id";
                $delete_stmt = $this->conn->prepare($delete_query);
                $delete_stmt->bindParam(":message_id", $message_id);
                
                if ($delete_stmt->execute()) {
                    return true;
                }
            }
        }
        
        return false;
    }
    
    // Compter le nombre total de messages échangés entre deux utilisateurs
    public function countTotalMessages($user1_id, $user2_id) {
        // Requête pour compter les messages
        $query = "SELECT COUNT(*) as total_count
                  FROM " . $this->table_name . "
                  WHERE (sender_id = :user1_id AND receiver_id = :user2_id)
                     OR (sender_id = :user2_id AND receiver_id = :user1_id)";
        
        // Préparer la requête
        $stmt = $this->conn->prepare($query);
        
        // Lier les valeurs
        $stmt->bindParam(":user1_id", $user1_id);
        $stmt->bindParam(":user2_id", $user2_id);
        
        // Exécuter la requête
        $stmt->execute();
        
        // Récupérer le résultat
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        
        return $row['total_count'];
    }
}
?>