import React, { useState } from 'react';
import { MessageCircle, Search, Phone, Video, Info, PlusCircle, Image, Paperclip, Send, Settings, LogOut } from 'lucide-react';

function App() {
  const [activeContact, setActiveContact] = useState(1);
  const [messageInput, setMessageInput] = useState('');
  const [messages, setMessages] = useState([
    { id: 1, senderId: 1, receiverId: 2, content: 'Bonjour Jane, comment vas-tu ?', time: '14:30', isRead: true },
    { id: 2, senderId: 2, receiverId: 1, content: 'Salut John, je vais bien merci ! Et toi ?', time: '14:32', isRead: true },
    { id: 3, senderId: 1, receiverId: 2, content: 'Très bien aussi, merci !', time: '14:35', isRead: true },
    { id: 4, senderId: 3, receiverId: 1, content: 'Salut John, as-tu terminé le projet ?', time: '16:45', isRead: false },
    { id: 5, senderId: 1, receiverId: 3, content: 'Pas encore, mais j\'y travaille !', time: '17:20', isRead: true },
  ]);

  const contacts = [
    { id: 1, firstName: 'John', lastName: 'Doe', isOnline: true, lastMessage: 'Très bien aussi, merci !', lastMessageTime: '14:35', unreadCount: 0, avatar: 'https://images.unsplash.com/photo-1570295999919-56ceb5ecca61?ixlib=rb-4.0.3&ixid=MnwxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8&auto=format&fit=crop&w=880&q=80' },
    { id: 2, firstName: 'Jane', lastName: 'Smith', isOnline: true, lastMessage: 'Salut John, je vais bien merci ! Et toi ?', lastMessageTime: '14:32', unreadCount: 2, avatar: 'https://images.unsplash.com/photo-1494790108377-be9c29b29330?ixlib=rb-4.0.3&ixid=MnwxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8&auto=format&fit=crop&w=687&q=80' },
    { id: 3, firstName: 'Alice', lastName: 'Johnson', isOnline: false, lastMessage: 'Pas encore, mais j\'y travaille !', lastMessageTime: '17:20', unreadCount: 0, avatar: 'https://images.unsplash.com/photo-1580489944761-15a19d654956?ixlib=rb-4.0.3&ixid=MnwxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8&auto=format&fit=crop&w=761&q=80' },
  ];

  const currentUser = {
    id: 1,
    firstName: 'John',
    lastName: 'Doe',
    avatar: 'https://images.unsplash.com/photo-1570295999919-56ceb5ecca61?ixlib=rb-4.0.3&ixid=MnwxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8&auto=format&fit=crop&w=880&q=80'
  };

  const selectedContact = contacts.find(contact => contact.id === activeContact);

  const filteredMessages = messages.filter(
    message => 
      (message.senderId === currentUser.id && message.receiverId === activeContact) || 
      (message.receiverId === currentUser.id && message.senderId === activeContact)
  );

  const handleSendMessage = (e) => {
    e.preventDefault();
    if (messageInput.trim() === '') return;

    const newMessage = {
      id: messages.length + 1,
      senderId: currentUser.id,
      receiverId: activeContact,
      content: messageInput,
      time: new Date().toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'}),
      isRead: false
    };

    setMessages([...messages, newMessage]);
    setMessageInput('');
  };

  return (
    <div className="min-h-screen bg-gray-100 flex flex-col items-center justify-center p-4">
      <h1 className="text-2xl font-bold mb-4 text-blue-600">Prévisualisation de Messenger Clone</h1>
      <div className="w-full max-w-6xl bg-white rounded-lg shadow-lg overflow-hidden">
        <div className="flex h-[600px]">
          {/* Sidebar */}
          <div className="w-[350px] border-r border-gray-200 flex flex-col">
            {/* User profile */}
            <div className="flex items-center p-4 border-b border-gray-200">
              <img 
                src={currentUser.avatar} 
                alt="Profile" 
                className="w-10 h-10 rounded-full object-cover mr-3"
              />
              <div className="flex-1">
                <h3 className="font-semibold text-gray-800">{currentUser.firstName} {currentUser.lastName}</h3>
                <div className="flex items-center text-xs text-gray-500">
                  <span className="w-2 h-2 bg-green-500 rounded-full mr-1"></span>
                  <span>En ligne</span>
                </div>
              </div>
              <div className="flex gap-3">
                <button className="text-gray-500 hover:text-blue-500">
                  <Settings size={18} />
                </button>
                <button className="text-gray-500 hover:text-blue-500">
                  <LogOut size={18} />
                </button>
              </div>
            </div>
            
            {/* Search box */}
            <div className="p-3 border-b border-gray-200 relative">
              <Search size={16} className="absolute left-6 top-1/2 transform -translate-y-1/2 text-gray-400" />
              <input 
                type="text" 
                placeholder="Rechercher..." 
                className="w-full py-2 pl-10 pr-4 bg-gray-100 rounded-full text-sm border-none focus:outline-none focus:ring-2 focus:ring-blue-500"
              />
            </div>
            
            {/* Contacts list */}
            <div className="flex-1 overflow-y-auto">
              {contacts.map(contact => (
                <div 
                  key={contact.id}
                  onClick={() => setActiveContact(contact.id)}
                  className={`flex items-center p-3 border-b border-gray-200 cursor-pointer ${activeContact === contact.id ? 'bg-gray-100' : 'hover:bg-gray-50'}`}
                >
                  <div className="relative mr-3">
                    <img 
                      src={contact.avatar} 
                      alt={contact.firstName} 
                      className="w-12 h-12 rounded-full object-cover"
                    />
                    {contact.isOnline && (
                      <span className="absolute bottom-0 right-0 w-3 h-3 bg-green-500 rounded-full border-2 border-white"></span>
                    )}
                  </div>
                  <div className="flex-1 min-w-0">
                    <h4 className="font-semibold text-gray-800 truncate">{contact.firstName} {contact.lastName}</h4>
                    <p className="text-sm text-gray-500 truncate">{contact.lastMessage}</p>
                  </div>
                  <div className="flex flex-col items-end ml-2">
                    <span className="text-xs text-gray-500 mb-1">{contact.lastMessageTime}</span>
                    {contact.unreadCount > 0 && (
                      <span className="bg-blue-500 text-white text-xs font-semibold w-5 h-5 flex items-center justify-center rounded-full">
                        {contact.unreadCount}
                      </span>
                    )}
                  </div>
                </div>
              ))}
            </div>
          </div>
          
          {/* Chat area */}
          <div className="flex-1 flex flex-col">
            {selectedContact ? (
              <>
                {/* Chat header */}
                <div className="flex items-center justify-between p-3 border-b border-gray-200">
                  <div className="flex items-center">
                    <img 
                      src={selectedContact.avatar} 
                      alt={selectedContact.firstName} 
                      className="w-10 h-10 rounded-full object-cover mr-3"
                    />
                    <div>
                      <h3 className="font-semibold text-gray-800">{selectedContact.firstName} {selectedContact.lastName}</h3>
                      <span className="text-xs text-gray-500">
                        {selectedContact.isOnline ? 'En ligne' : 'Hors ligne'}
                      </span>
                    </div>
                  </div>
                  <div className="flex gap-3">
                    <button className="text-blue-500 hover:text-blue-700">
                      <Phone size={18} />
                    </button>
                    <button className="text-blue-500 hover:text-blue-700">
                      <Video size={18} />
                    </button>
                    <button className="text-blue-500 hover:text-blue-700">
                      <Info size={18} />
                    </button>
                  </div>
                </div>
                
                {/* Messages */}
                <div className="flex-1 p-4 overflow-y-auto bg-white">
                  {filteredMessages.map(message => (
                    <div 
                      key={message.id}
                      className={`flex mb-4 ${message.senderId === currentUser.id ? 'justify-end' : 'justify-start'}`}
                    >
                      <div 
                        className={`max-w-[60%] px-4 py-2 rounded-lg ${
                          message.senderId === currentUser.id 
                            ? 'bg-blue-500 text-white rounded-tr-sm' 
                            : 'bg-gray-100 rounded-tl-sm'
                        }`}
                      >
                        {message.content}
                        <span 
                          className={`block text-xs mt-1 ${
                            message.senderId === currentUser.id 
                              ? 'text-blue-100 text-right' 
                              : 'text-gray-500'
                          }`}
                        >
                          {message.time}
                        </span>
                      </div>
                    </div>
                  ))}
                </div>
                
                {/* Input area */}
                <div className="flex items-center p-3 border-t border-gray-200">
                  <div className="flex gap-3 mr-2">
                    <button className="text-blue-500 hover:text-blue-700">
                      <PlusCircle size={18} />
                    </button>
                    <button className="text-blue-500 hover:text-blue-700">
                      <Image size={18} />
                    </button>
                    <button className="text-blue-500 hover:text-blue-700">
                      <Paperclip size={18} />
                    </button>
                  </div>
                  <form onSubmit={handleSendMessage} className="flex flex-1">
                    <input 
                      type="text" 
                      placeholder="Écrivez un message..." 
                      className="flex-1 py-2 px-4 border border-gray-200 rounded-full text-sm focus:outline-none focus:ring-2 focus:ring-blue-500"
                      value={messageInput}
                      onChange={(e) => setMessageInput(e.target.value)}
                    />
                    <button 
                      type="submit" 
                      className="ml-2 text-blue-500 hover:text-blue-700"
                    >
                      <Send size={18} />
                    </button>
                  </form>
                </div>
              </>
            ) : (
              <div className="flex-1 flex items-center justify-center">
                <div className="text-center p-5">
                  <MessageCircle size={50} className="mx-auto mb-4 text-blue-500" />
                  <h2 className="text-xl font-semibold mb-2">Bienvenue sur Messenger</h2>
                  <p className="text-gray-500">Sélectionnez un contact pour commencer à discuter</p>
                </div>
              </div>
            )}
          </div>
        </div>
      </div>
      
      <div className="mt-4 text-gray-600 text-sm">
        <p className="font-semibold">Informations sur la prévisualisation:</p>
        <ul className="list-disc pl-5 mt-2">
          <li>Cette interface est une simulation du clone de Messenger</li>
          <li>Vous pouvez cliquer sur les contacts pour changer de conversation</li>
          <li>Vous pouvez envoyer des messages dans la conversation active</li>
          <li>Cette prévisualisation montre l'apparence et les fonctionnalités de base du site</li>
        </ul>
      </div>
    </div>
  );
}

export default App;