import React, { useState, useEffect } from 'react';
import { Save, Trash2, Plus, FileText } from 'lucide-react';

function App() {
  const [notes, setNotes] = useState<{ id: string; title: string; content: string; date: string }[]>([]);
  const [currentNote, setCurrentNote] = useState<{ id: string; title: string; content: string; date: string } | null>(null);
  const [searchTerm, setSearchTerm] = useState('');

  // Charger les notes depuis le stockage local au démarrage
  useEffect(() => {
    const savedNotes = localStorage.getItem('notes');
    if (savedNotes) {
      setNotes(JSON.parse(savedNotes));
    } else {
      // Créer une note par défaut si aucune n'existe
      const defaultNote = {
        id: Date.now().toString(),
        title: 'Bienvenue dans votre application de notes',
        content: 'Commencez à écrire vos notes ici. Vous pouvez créer, modifier et supprimer des notes facilement.',
        date: new Date().toLocaleString()
      };
      setNotes([defaultNote]);
      setCurrentNote(defaultNote);
      localStorage.setItem('notes', JSON.stringify([defaultNote]));
    }
  }, []);

  // Sauvegarder les notes dans le stockage local à chaque modification
  useEffect(() => {
    if (notes.length > 0) {
      localStorage.setItem('notes', JSON.stringify(notes));
    }
  }, [notes]);

  // Créer une nouvelle note
  const createNewNote = () => {
    const newNote = {
      id: Date.now().toString(),
      title: 'Nouvelle note',
      content: '',
      date: new Date().toLocaleString()
    };
    setNotes([newNote, ...notes]);
    setCurrentNote(newNote);
  };

  // Sauvegarder la note actuelle
  const saveCurrentNote = () => {
    if (!currentNote) return;
    
    const updatedNotes = notes.map(note => 
      note.id === currentNote.id 
        ? {...currentNote, date: new Date().toLocaleString()} 
        : note
    );
    
    setNotes(updatedNotes);
  };

  // Supprimer une note
  const deleteNote = (id: string) => {
    const updatedNotes = notes.filter(note => note.id !== id);
    setNotes(updatedNotes);
    
    if (currentNote && currentNote.id === id) {
      setCurrentNote(updatedNotes.length > 0 ? updatedNotes[0] : null);
    }
  };

  // Filtrer les notes en fonction du terme de recherche
  const filteredNotes = notes.filter(note => 
    note.title.toLowerCase().includes(searchTerm.toLowerCase()) || 
    note.content.toLowerCase().includes(searchTerm.toLowerCase())
  );

  return (
    <div className="min-h-screen bg-gray-100 flex">
      {/* Sidebar */}
      <div className="w-64 bg-white border-r border-gray-200 flex flex-col">
        <div className="p-4 border-b border-gray-200">
          <div className="flex items-center justify-between mb-4">
            <h1 className="text-xl font-bold text-gray-800">Mes Notes</h1>
            <button 
              onClick={createNewNote}
              className="p-2 rounded-full hover:bg-gray-100"
              title="Nouvelle note"
            >
              <Plus size={20} />
            </button>
          </div>
          <div className="relative">
            <input
              type="text"
              placeholder="Rechercher..."
              className="w-full p-2 border border-gray-300 rounded-md"
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
            />
          </div>
        </div>
        <div className="flex-1 overflow-y-auto">
          {filteredNotes.map(note => (
            <div 
              key={note.id} 
              className={`p-3 border-b border-gray-200 cursor-pointer flex items-start justify-between ${currentNote && currentNote.id === note.id ? 'bg-blue-50' : 'hover:bg-gray-50'}`}
              onClick={() => setCurrentNote(note)}
            >
              <div className="flex-1 min-w-0">
                <div className="flex items-center">
                  <FileText size={16} className="text-gray-500 mr-2" />
                  <h3 className="text-sm font-medium text-gray-800 truncate">{note.title}</h3>
                </div>
                <p className="text-xs text-gray-500 mt-1 truncate">{note.content.substring(0, 50)}{note.content.length > 50 ? '...' : ''}</p>
                <p className="text-xs text-gray-400 mt-1">{note.date}</p>
              </div>
              <button 
                onClick={(e) => {
                  e.stopPropagation();
                  deleteNote(note.id);
                }}
                className="p-1 rounded-full hover:bg-gray-200 text-gray-500 hover:text-red-500"
                title="Supprimer"
              >
                <Trash2 size={16} />
              </button>
            </div>
          ))}
          {filteredNotes.length === 0 && (
            <div className="p-4 text-center text-gray-500">
              Aucune note trouvée
            </div>
          )}
        </div>
      </div>

      {/* Main content */}
      <div className="flex-1 flex flex-col">
        {currentNote ? (
          <>
            <div className="p-4 border-b border-gray-200 flex justify-between items-center bg-white">
              <input
                type="text"
                className="text-xl font-bold w-full bg-transparent border-none focus:outline-none focus:ring-0"
                value={currentNote.title}
                onChange={(e) => setCurrentNote({...currentNote, title: e.target.value})}
                placeholder="Titre de la note"
              />
              <button 
                onClick={saveCurrentNote}
                className="ml-2 p-2 bg-blue-500 text-white rounded-md hover:bg-blue-600 flex items-center"
              >
                <Save size={16} className="mr-1" />
                Enregistrer
              </button>
            </div>
            <div className="flex-1 p-4 overflow-auto">
              <textarea
                className="w-full h-full p-2 border border-gray-300 rounded-md resize-none focus:outline-none focus:ring-1 focus:ring-blue-500"
                value={currentNote.content}
                onChange={(e) => setCurrentNote({...currentNote, content: e.target.value})}
                placeholder="Contenu de votre note..."
              />
            </div>
          </>
        ) : (
          <div className="flex-1 flex items-center justify-center bg-gray-50">
            <div className="text-center">
              <FileText size={48} className="mx-auto text-gray-400 mb-4" />
              <h2 className="text-xl font-medium text-gray-600 mb-2">Aucune note sélectionnée</h2>
              <p className="text-gray-500 mb-4">Sélectionnez une note existante ou créez-en une nouvelle</p>
              <button 
                onClick={createNewNote}
                className="px-4 py-2 bg-blue-500 text-white rounded-md hover:bg-blue-600 flex items-center mx-auto"
              >
                <Plus size={16} className="mr-1" />
                Nouvelle note
              </button>
            </div>
          </div>
        )}
      </div>
    </div>
  );
}

export default App;